from sqlalchemy import (
    Column,
    Integer,
    String,
    JSON,
    Boolean,
    DECIMAL,
    TIMESTAMP,
    Computed,
    Index,
    ForeignKey,
)
from sqlalchemy.orm import relationship
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.sql import func

Base = declarative_base()


class User(Base):
    __tablename__ = "users"

    id = Column(Integer, primary_key=True)
    gender = Column(String(50))  # e.g., 'Male', 'Female', 'Other'

    ai_logs = relationship("AILog", back_populates="user")


class AILog(Base):
    __tablename__ = "ai_logs"

    id = Column(Integer, primary_key=True, autoincrement=True)

    # User Context
    user_id = Column(Integer, ForeignKey("users.id"), nullable=False, index=True)
    thread_id = Column(String(80), nullable=False, index=True)

    user = relationship("User", back_populates="ai_logs")

    # Query Details
    raw_query = Column(String(512), nullable=False)
    parsed_food_items = Column(JSON)
    parsed_restaurant = Column(String(255), index=True)
    parsed_avoid_items = Column(JSON)
    parsed_deals_only = Column(Boolean, default=False)
    parsed_low_to_high = Column(Boolean, default=False)
    parsed_sort_by_price = Column(Boolean, default=False)

    # Results & Performance
    results_found = Column(Integer, default=0)
    success = Column(
        Boolean, Computed("results_found > 0", persisted=True), index=True
    )
    top_restaurant = Column(String(255))
    response_time_ms = Column(Integer)

    # Geographic Context
    user_lat = Column(DECIMAL(10, 8))
    user_long = Column(DECIMAL(11, 8))
    country = Column(String(100), index=True)

    # Time Context
    created_at = Column(
        TIMESTAMP, server_default=func.now(), nullable=False, index=True
    )

    __table_args__ = (
        Index("idx_user_id", "user_id"),
        Index("idx_thread_id", "thread_id"),
        Index("idx_created_at", "created_at"),
        Index("idx_country", "country"),
        Index("idx_parsed_restaurant", "parsed_restaurant"),
        Index("idx_success", "success"),
    )

    def __repr__(self):
        return f"<AILog(id={self.id}, user_id={self.user_id}, raw_query='{self.raw_query}')>"
