from fastapi import APIRouter, Depends, Query
from sqlalchemy.ext.asyncio import AsyncSession
from services.analytics_service import AnalyticsService, get_analytics_service
from database.connection import get_db
from pydantic import BaseModel
from typing import List, Dict, Any
from datetime import date
from routers.overview import get_date_query_params

router = APIRouter(prefix="/food-insights", tags=["Food & Cuisine Insights"])


# Pydantic Models
class TopItemsResponse(BaseModel):
    data: List[Dict[str, Any]]
    start_date: date
    end_date: date


class CountryDistributionItem(BaseModel):
    country: str
    count: int
    percentage: float


class ZeroResultQueryItem(BaseModel):
    query: str
    attempts: int
    country_distribution: List[CountryDistributionItem]


class ZeroResultResponse(BaseModel):
    data: List[ZeroResultQueryItem]
    start_date: date
    end_date: date


@router.get("/top-food-items", response_model=TopItemsResponse)
async def get_top_food_items(
    dates: tuple[date, date] = Depends(get_date_query_params),
    limit: int = Query(10, ge=1, le=100),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the most frequently searched food items.
    """
    start_date, end_date = dates
    top_foods = await analytics_service.get_top_json_items(
        db, start_date, end_date, "parsed_food_items", limit
    )
    return {"data": top_foods, "start_date": start_date, "end_date": end_date}


@router.get("/top-avoid-items", response_model=TopItemsResponse)
async def get_top_avoid_items(
    dates: tuple[date, date] = Depends(get_date_query_params),
    limit: int = Query(10, ge=1, le=100),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the most frequently avoided food items.
    """
    start_date, end_date = dates
    top_avoid_items = await analytics_service.get_top_json_items(
        db, start_date, end_date, "parsed_avoid_items", limit
    )
    return {"data": top_avoid_items, "start_date": start_date, "end_date": end_date}


@router.get("/zero-result-queries", response_model=ZeroResultResponse)
async def get_zero_result_queries(
    dates: tuple[date, date] = Depends(get_date_query_params),
    limit: int = Query(10, ge=1, le=100),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the most common queries that resulted in zero results.
    """
    start_date, end_date = dates
    queries = await analytics_service.get_zero_result_queries(
        db, start_date, end_date, limit
    )
    return {"data": queries, "start_date": start_date, "end_date": end_date}
