from fastapi import APIRouter, Depends, Query
from sqlalchemy.ext.asyncio import AsyncSession
from services.analytics_service import AnalyticsService, get_analytics_service
from database.connection import get_db
from pydantic import BaseModel
from typing import List, Dict, Any
from datetime import date
from routers.overview import get_date_query_params

router = APIRouter(prefix="/geographic-insights", tags=["Geographic Insights"])


# Pydantic Models
class SearchesByCountryResponse(BaseModel):
    data: List[Dict[str, Any]]
    start_date: date
    end_date: date


class SuccessRateByCountryResponse(BaseModel):
    data: List[Dict[str, Any]]
    start_date: date
    end_date: date


@router.get("/searches-by-country", response_model=SearchesByCountryResponse)
async def get_searches_by_country(
    dates: tuple[date, date] = Depends(get_date_query_params),
    limit: int = Query(10, ge=1, le=100),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the number of searches aggregated by country.
    """
    start_date, end_date = dates
    searches = await analytics_service.get_searches_by_country(
        db, start_date, end_date, limit
    )
    return {"data": searches, "start_date": start_date, "end_date": end_date}


@router.get("/success-rate-by-country", response_model=SuccessRateByCountryResponse)
async def get_success_rate_by_country(
    dates: tuple[date, date] = Depends(get_date_query_params),
    limit: int = Query(10, ge=1, le=100),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the search success rate aggregated by country.
    """
    start_date, end_date = dates
    rates = await analytics_service.get_success_rate_by_country(
        db, start_date, end_date, limit
    )
    return {"data": rates, "start_date": start_date, "end_date": end_date}
