import asyncio
from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy.ext.asyncio import AsyncSession
from services.analytics_service import AnalyticsService, get_analytics_service
from database.connection import get_db
from pydantic import BaseModel
from typing import Dict, List, Any
from datetime import date, timedelta

router = APIRouter(prefix="/overview", tags=["Overview Reports"])


# Pydantic Models for Response Schemas
class TotalSearchesResponse(BaseModel):
    total_searches: int
    start_date: date
    end_date: date


class UniqueUsersResponse(BaseModel):
    unique_users: int
    start_date: date
    end_date: date


class SuccessRateResponse(BaseModel):
    success_rate_percent: float
    start_date: date
    end_date: date


class SearchesTrendResponse(BaseModel):
    trend_data: Dict[date, int]
    start_date: date
    end_date: date


from pydantic import BaseModel
from typing import Dict, List, Any, Optional

# ... existing code ...

class OverviewSummaryResponse(BaseModel):
    total_searches: int
    unique_users: int
    success_rate_percent: float
    trend_data: Dict[date, int]
    top_countries: List[Dict[str, Any]]
    gender_distribution: List[Dict[str, Any]]
    start_date: date
    end_date: date

def get_date_query_params(
    start_date: date = Query(None), end_date: date = Query(None)
):
    """Dependency to get and validate date query parameters."""
    if end_date is None:
        end_date = date.today()
    if start_date is None:
        start_date = end_date - timedelta(days=30)

    if start_date > end_date:
        raise HTTPException(
            status_code=400,
            detail="start_date cannot be after end_date.",
        )
    return start_date, end_date


@router.get("/summary", response_model=OverviewSummaryResponse)
async def get_overview_summary(
    dates: tuple[date, date] = Depends(get_date_query_params),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get a combined summary of all overview metrics for the dashboard homepage.
    """
    start_date, end_date = dates

    # Perform the queries sequentially for safety and clarity.
    kpis = await analytics_service.get_overview_kpis(db, start_date, end_date)
    trend_data = await analytics_service.get_searches_trend(db, start_date, end_date)
    top_countries = await analytics_service.get_searches_by_country(
        db, start_date, end_date, limit=3
    )
    gender_distribution = await analytics_service.get_gender_distribution(
        db, start_date, end_date
    )

    return {
        "total_searches": kpis["total_searches"],
        "unique_users": kpis["unique_users"],
        "success_rate_percent": kpis["success_rate_percent"],
        "trend_data": trend_data,
        "top_countries": top_countries,
        "gender_distribution": gender_distribution,
        "start_date": start_date,
        "end_date": end_date,
    }


@router.get("/total-searches", response_model=TotalSearchesResponse)
async def get_total_searches_endpoint(
    dates: tuple[date, date] = Depends(get_date_query_params),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the total number of searches within a specified date range.
    Defaults to the last 30 days if no dates are provided.
    """
    start_date, end_date = dates
    total_searches = await analytics_service.get_total_searches(
        db, start_date, end_date
    )
    return {
        "total_searches": total_searches,
        "start_date": start_date,
        "end_date": end_date,
    }


@router.get("/unique-users", response_model=UniqueUsersResponse)
async def get_unique_users_endpoint(
    dates: tuple[date, date] = Depends(get_date_query_params),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the number of unique users who performed a search in the date range.
    Defaults to the last 30 days.
    """
    start_date, end_date = dates
    unique_users = await analytics_service.get_unique_users(db, start_date, end_date)
    return {
        "unique_users": unique_users,
        "start_date": start_date,
        "end_date": end_date,
    }


@router.get("/success-rate", response_model=SuccessRateResponse)
async def get_success_rate_endpoint(
    dates: tuple[date, date] = Depends(get_date_query_params),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the search success rate (searches with results / total searches).
    Defaults to the last 30 days.
    """
    start_date, end_date = dates
    success_rate = await analytics_service.get_success_rate(db, start_date, end_date)
    return {
        "success_rate_percent": success_rate,
        "start_date": start_date,
        "end_date": end_date,
    }


@router.get("/searches-trend", response_model=SearchesTrendResponse)
async def get_searches_trend_endpoint(
    dates: tuple[date, date] = Depends(get_date_query_params),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the daily trend of searches over a date range.
    Defaults to the last 30 days.
    """
    start_date, end_date = dates
    trend_data = await analytics_service.get_searches_trend(db, start_date, end_date)
    return {
        "trend_data": trend_data,
        "start_date": start_date,
        "end_date": end_date,
}