from fastapi import APIRouter, Depends, Query
from sqlalchemy.ext.asyncio import AsyncSession
from services.analytics_service import AnalyticsService, get_analytics_service
from database.connection import get_db
from pydantic import BaseModel
from typing import List, Dict, Any
from datetime import date
from routers.overview import get_date_query_params

router = APIRouter(
    prefix="/restaurant-insights", tags=["Restaurant Insights"]
)


# Pydantic Models
class TopRestaurantsResponse(BaseModel):
    data: List[Dict[str, Any]]
    start_date: date
    end_date: date


@router.get("/most-searched", response_model=TopRestaurantsResponse)
async def get_most_searched_restaurants(
    dates: tuple[date, date] = Depends(get_date_query_params),
    limit: int = Query(10, ge=1, le=100),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the most searched-for restaurants.
    """
    start_date, end_date = dates
    restaurants = await analytics_service.get_most_common_restaurants(
        db, start_date, end_date, "parsed_restaurant", limit
    )
    return {"data": restaurants, "start_date": start_date, "end_date": end_date}


@router.get("/most-returned-as-top", response_model=TopRestaurantsResponse)
async def get_most_returned_as_top_restaurant(
    dates: tuple[date, date] = Depends(get_date_query_params),
    limit: int = Query(10, ge=1, le=100),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the restaurants most frequently returned as the top result.
    """
    start_date, end_date = dates
    restaurants = await analytics_service.get_most_common_restaurants(
        db, start_date, end_date, "top_restaurant", limit
    )
    return {"data": restaurants, "start_date": start_date, "end_date": end_date}
