from fastapi import APIRouter, Depends
from sqlalchemy.ext.asyncio import AsyncSession
from services.analytics_service import AnalyticsService, get_analytics_service
from database.connection import get_db
from pydantic import BaseModel
from typing import List, Dict, Any
from datetime import date
from routers.overview import get_date_query_params

router = APIRouter(prefix="/user-insights", tags=["User Insights"])


# Pydantic Models
class GenderDistributionResponse(BaseModel):
    data: List[Dict[str, Any]]
    start_date: date
    end_date: date


@router.get("/gender-distribution", response_model=GenderDistributionResponse)
async def get_gender_distribution(
    dates: tuple[date, date] = Depends(get_date_query_params),
    db: AsyncSession = Depends(get_db),
    analytics_service: AnalyticsService = Depends(get_analytics_service),
):
    """
    Get the distribution of searches by user gender.
    """
    start_date, end_date = dates
    distribution = await analytics_service.get_gender_distribution(
        db, start_date, end_date
    )
    return {"data": distribution, "start_date": start_date, "end_date": end_date}
